//
//  GSClass.h
//  GlyphsCore
//
//  Created by Georg Seifert on 25.10.10.
//  Copyright 2010 schriftgestaltung.de. All rights reserved.
//

#import <Cocoa/Cocoa.h>
#import <GlyphsCore/GSBase.h>

@class GSFont;
/**
 The class represents the OpenType classes
 */

NS_ASSUME_NONNULL_BEGIN

@interface GSClass : NSObject <NSCoding, NSCopying> {
	GSFont *__unsafe_unretained _parent;
	NSString *_name;
	NSString *_code;
	BOOL _automatic;
	BOOL _disabled;
	NSString *_notes;
	NSString *_predicate;
	NSArray *_errors;
}

- (instancetype)initWithDict:(NSDictionary *)dict format:(GSFormatVersion)formatVersion;

- (BOOL)isEqualToClass:(GSClass *)other;

// compares the code independent of whitespace
- (BOOL)isEqualToClassLenient:(GSClass *)other;
/**
 a reference to the font containing the class
 */
@property (nonatomic, unsafe_unretained, nullable) GSFont *parent;

/**
 The name of the class
 */
@property (nonatomic, copy) NSString *name;

- (BOOL)validateName:(id _Nonnull *_Nonnull)ioValue error:(NSError **)error;

- (NSString *)identifier;

/**
 The content of the class

 Normally glyph names seperated my spaces.
 */
@property (strong, nonatomic) NSString *code;

@property (strong, nonatomic) NSAttributedString *attributedCode;

- (NSString *)saveCode;
/**
 if the class is replaced on automatic feature generation.
 */
@property (nonatomic) BOOL automatic;

@property (nonatomic, strong) NSString *predicate;

/** If True, the feature will not be exported. */
@property (nonatomic) BOOL disabled;

/**
 Some notes for the class.
 */
@property (nonatomic, strong) NSString *notes;
//@property (nonatomic, unsafe_unretained) NSDictionary *classDict;

/** Returns the content of the object to store it in pList.

 This is used to store the data in the .glyphs file.
 @param format the version of the dict
 */
- (NSDictionary *)propertyListValueFormat:(GSFormatVersion)format;

@property (nonatomic, strong, nullable) NSArray<NSError *> *errors;

@property (nonatomic, assign) NSInteger errorType;

@property (nonatomic, strong, nullable) NSString *errorTooltip;

@property (nonatomic, readonly, nullable) NSImage *errorIcon;

#ifndef GLYPHS_VIEWER

- (BOOL)saveToFile:(FILE *)file format:(GSFormatVersion)formatVersion error:(NSError **)error;

/**
 Triggers the regeneration of the class.
 */
- (void)update;

/// of the featureGenerator can generate the class
@property (nonatomic, readonly) BOOL canBeAutomated;

#pragma mark TempData
/**
 a  dictionary that stores data. It will not be written to disk.
 */
@property (nonatomic, strong, nullable) NSDictionary *tempData;

/**
 Adds key/value to tempData. Pass nil as value to remove previous set data

 @param value and object or nil
 @param key the key
 */
- (void)setTempData:(nullable id)value forKey:(nonnull NSString *)key;

/**
 return value for key in tempData

 @param key the key
 @return a value or nil
 */
- (nullable id)tempDataForKey:(nonnull NSString *)key;

// This is only to be used during export to mark features to that they are not visible in the UI. This is to keep the indexes in the `### feature:X:yyyy ###` in sync
@property (assign) BOOL isVisibleInUI;

#endif
@end

NS_ASSUME_NONNULL_END
